/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { deepClone } from 'common';
import { BehaviorSubject, of } from 'rxjs';
import { shareReplay, tap } from 'rxjs/operators';
import { RecordStore } from '../../../../store/record/record.store';
import { UntypedFormGroup } from '@angular/forms';
import { signal } from "@angular/core";
const initialState = {
    id: '',
    type: '',
    module: '',
    attributes: {}
};
export class SavedFilterRecordStore extends RecordStore {
    constructor(definitions$, metadata$, recordSaveGQL, recordFetchGQL, message, recordManager, recordMappers, fieldManager, language) {
        super(definitions$, metadata$, recordSaveGQL, recordFetchGQL, message, recordManager, recordMappers);
        this.definitions$ = definitions$;
        this.metadata$ = metadata$;
        this.recordSaveGQL = recordSaveGQL;
        this.recordFetchGQL = recordFetchGQL;
        this.message = message;
        this.recordManager = recordManager;
        this.recordMappers = recordMappers;
        this.fieldManager = fieldManager;
        this.language = language;
        this.internalState = deepClone(initialState);
        this.stagingState = deepClone(initialState);
        this.store = new BehaviorSubject(this.internalState);
        this.staging = new BehaviorSubject(this.stagingState);
        this.searchFields = {};
        this.listColumns = [];
        this.state$ = this.store.asObservable().pipe(tap(record => {
            this.updateStaging(record);
        }));
        this.staging$ = this.staging.asObservable();
    }
    /**
     * Get search fields metadata
     * @returns SearchMetaFieldMap
     */
    getSearchFields() {
        return this.searchFields;
    }
    /**
     * Set search fields metadata
     * @param {object} searchFields SearchMetaFieldMap
     */
    setSearchFields(searchFields) {
        this.searchFields = searchFields;
    }
    /**
     * Get list fields metadata
     * @returns SearchMetaFieldMap
     */
    getListColumns() {
        return this.listColumns;
    }
    /**
     * Set list fields metadata
     * @param {object} listColumns SearchMetaFieldMap
     */
    setListColumns(listColumns) {
        this.listColumns = listColumns;
    }
    /**
     * Get record
     *
     * @returns {object} Record
     */
    getBaseRecord() {
        if (!this.stagingState) {
            return null;
        }
        this.mapStagingFields();
        return deepClone({
            id: this.stagingState.id,
            type: this.stagingState.type,
            module: this.stagingState.module,
            key: this.stagingState.key,
            searchModule: this.stagingState.searchModule,
            criteria: this.stagingState.criteria,
            attributes: this.stagingState.attributes,
        });
    }
    /**
     * Extract base record
     *
     * @returns {object} Record
     */
    extractBaseRecord(record) {
        if (!record) {
            return null;
        }
        let criteria = record.criteria ?? {};
        if (Array.isArray(criteria) && !criteria.length) {
            criteria = {};
        }
        return deepClone({
            id: record.id,
            type: record.type,
            module: record.module,
            key: record.key,
            searchModule: record.searchModule,
            criteria: criteria,
            attributes: record.attributes,
        });
    }
    /**
     * Init record fields
     *
     * @param {object} record Record
     */
    initRecord(record) {
        if (this.metadata) {
            record.metadata = this.metadata;
        }
        if (!record?.validationTriggered) {
            record.validationTriggered = signal(false);
        }
        record.attributes = record.attributes || {};
        record.attributes.search_module = record.searchModule;
        const filters = record?.attributes?.contents?.filters ?? {};
        record.attributes.contents = record.attributes.contents || { filters: {} };
        if (Array.isArray(filters) && !filters.length) {
            record.attributes.contents.filters = {};
        }
        else {
            record.attributes.contents.filters = filters;
        }
        record.criteria = this.getCriteria(record);
        this.initCriteriaFields(record, this.getSearchFields());
        if (record.module && this.definitions && this.definitions.length > 0) {
            record.fields = this.recordManager.initFields(record, this.definitions);
        }
        this.initOrderByOptions(record);
    }
    /**
     * Init Order by options using list view columns set as default
     * @param record
     */
    initOrderByOptions(record) {
        if (!record.fields || !record.fields.orderBy) {
            return;
        }
        record.fields.orderBy.metadata = record.fields.orderBy.metadata || {};
        const options = [];
        this.getListColumns().forEach(column => {
            if (!column.default || column.default !== true) {
                return;
            }
            const labelKey = column.label || column.fieldDefinition.vname || '';
            const label = this.language.getFieldLabel(labelKey, record.searchModule);
            options.push({
                value: column.fieldDefinition.name || column.name,
                label
            });
        });
        record.fields.orderBy.metadata.options$ = of(options).pipe(shareReplay());
    }
    /**
     * Get criteria from filter
     * @param filter
     */
    getCriteria(filter) {
        if (!filter || !filter.criteria) {
            return { filters: {} };
        }
        if (!filter.criteria.filters) {
            return { ...filter.criteria, filters: {} };
        }
        if (Array.isArray(filter.criteria.filters) && !filter.criteria.filters.length) {
            return { ...filter.criteria, filters: {} };
        }
        return deepClone(filter.criteria);
    }
    /**
     * Initialize criteria fields
     *
     * @param {object} record to use
     * @param {object} searchFields to use
     */
    initCriteriaFields(record, searchFields) {
        record.criteriaFields = record.criteriaFields || {};
        if (!record.criteriaFormGroup) {
            record.criteriaFormGroup = new UntypedFormGroup({});
        }
        if (!searchFields) {
            return;
        }
        Object.keys(searchFields).forEach(key => {
            this.buildField(record, searchFields[key]);
        });
    }
    /**
     * Build filter field according to Field interface
     *
     * @param {object} record SavedFilter
     * @param {object} fieldMeta to use
     */
    buildField(record, fieldMeta) {
        const fieldName = fieldMeta.name;
        const type = fieldMeta.type;
        const definition = {
            name: fieldMeta.name,
            label: fieldMeta.label,
            vardefBased: fieldMeta?.vardefBased ?? false,
            readonly: fieldMeta?.readonly ?? false,
            display: fieldMeta?.display ?? '',
            type,
            fieldDefinition: {}
        };
        if (fieldMeta.fieldDefinition) {
            definition.fieldDefinition = fieldMeta.fieldDefinition;
        }
        if (type === 'bool' || type === 'boolean') {
            definition.fieldDefinition.options = 'dom_int_bool';
        }
        this.fieldManager.addFilterField(record, definition, this.language);
    }
}
//# sourceMappingURL=data:application/json;base64,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