/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Component, computed, inject, Input, signal } from '@angular/core';
import { isVoid } from 'common';
import { BehaviorSubject } from 'rxjs';
import { DataTypeFormatter } from '../../services/formatters/data-type.formatter.service';
import { debounceTime } from 'rxjs/operators';
import { FieldLogicManager } from '../field-logic/field-logic.manager';
import { FieldLogicDisplayManager } from '../field-logic-display/field-logic-display.manager';
import { isEqual } from "lodash-es";
import { FieldHandlerRegistry } from "../../services/record/field/handler/field-handler.registry";
import * as i0 from "@angular/core";
import * as i1 from "../../services/formatters/data-type.formatter.service";
import * as i2 from "../field-logic/field-logic.manager";
import * as i3 from "../field-logic-display/field-logic-display.manager";
class BaseFieldComponent {
    get mode() {
        return this._mode;
    }
    set mode(value) {
        this._mode = value;
        this.modeState.next(this._mode);
    }
    constructor(typeFormatter, logic, logicDisplay) {
        this.typeFormatter = typeFormatter;
        this.logic = logic;
        this.logicDisplay = logicDisplay;
        this.originalMode = '';
        this.klass = null;
        this._mode = '';
        this.dependentFields = {};
        this.dependentAttributes = [];
        this.subs = [];
        this.validateOnlyOnSubmit = false;
        this.isInvalid = signal(false);
        this.modeState = new BehaviorSubject('');
        this.mode$ = this.modeState.asObservable();
        this.fieldHandlerRegistry = inject(FieldHandlerRegistry);
    }
    ngOnInit() {
        this.baseInit();
        if (!this.originalMode) {
            this.originalMode = this.mode;
        }
        const defaultValueModes = this?.field?.defaultValueModes ?? [];
        if (defaultValueModes.includes(this.originalMode)) {
            const fieldHandler = this.fieldHandlerRegistry.get(this.record.module, this.field.type);
            fieldHandler.initDefaultValue(this.field, this.record);
        }
    }
    ngOnDestroy() {
        this.unsubscribeAll();
    }
    baseInit() {
        this.initDependencyHandlers();
        this.validateOnlyOnSubmit = this.record?.metadata?.validateOnlyOnSubmit;
        if (this.record?.validationTriggered) {
            this.isInvalid = computed(() => {
                if (this.record?.metadata?.validateOnlyOnSubmit && this.record?.validationTriggered() && this.field.formControl?.invalid) {
                    return true;
                }
                return false;
            });
        }
    }
    /**
     * Calculate and init dependency handlers
     */
    initDependencyHandlers() {
        if (!this.record) {
            return;
        }
        const fieldKeys = (this.record.fields && Object.keys(this.record.fields)) || [];
        if (fieldKeys.length > 1) {
            this.calculateDependentFields(fieldKeys);
            this.field.previousValue = this.field.value;
            if ((this.dependentFields && Object.keys(this.dependentFields).length) || this.dependentAttributes.length) {
                Object.keys(this.dependentFields).forEach(fieldKey => {
                    const field = this.record.fields[fieldKey] || null;
                    if (!field) {
                        return;
                    }
                    const types = this.dependentFields[fieldKey].type ?? [];
                    if (types.includes('logic')) {
                        this.logic.runLogic(field, this.originalMode, this.record, 'onFieldInitialize');
                    }
                    if (types.includes('displayLogic')) {
                        this.logicDisplay.runAll(field, this.record, this.originalMode);
                    }
                });
            }
            if (this.field.valueChanges$ && ((this.dependentFields && Object.keys(this.dependentFields).length) || this.dependentAttributes.length)) {
                this.subs.push(this.field.valueChanges$.pipe(debounceTime(500)).subscribe((data) => {
                    Object.keys(this.dependentFields).forEach(fieldKey => {
                        const dependentField = this.dependentFields[fieldKey];
                        const field = this.record.fields[fieldKey] || null;
                        if (!field) {
                            return;
                        }
                        if (this.field.previousValue != data.value) {
                            const types = dependentField.type ?? [];
                            if (types.includes('logic')) {
                                this.logic.runLogic(field, this.originalMode, this.record, 'onValueChange');
                            }
                            if (types.includes('displayLogic')) {
                                this.logicDisplay.runAll(field, this.record, this.originalMode);
                            }
                        }
                    });
                    this.field.previousValue = data.value;
                    this.dependentAttributes.forEach(dependency => {
                        const field = this.record.fields[dependency.field] || {};
                        const attribute = (field && field.attributes && field.attributes[dependency.attribute]) || null;
                        if (!attribute) {
                            return;
                        }
                        this.logic.runLogic(attribute, this.mode, this.record, 'onValueChange');
                    });
                }));
            }
        }
    }
    /**
     * Calculate dependent fields
     * @param {array} fieldKeys
     */
    calculateDependentFields(fieldKeys) {
        fieldKeys.forEach(key => {
            if (this.field.source === 'field' || this.field.source === 'groupField') {
                this.addFieldDependency(key, this.dependentFields, this.dependentAttributes);
                return;
            }
            if (this.field.source === 'attribute') {
                this.addAttributeDependency(key, this.dependentFields, this.dependentAttributes);
                return;
            }
        });
    }
    /**
     * Add field dependency
     * @param {string} fieldKey
     * @param {array} dependentFields
     * @param {object} dependentAttributes
     */
    addFieldDependency(fieldKey, dependentFields, dependentAttributes) {
        const field = this.record.fields[fieldKey];
        const name = this.field.name || this.field.definition.name || '';
        if (fieldKey === name || !field) {
            return;
        }
        if (field.fieldDependencies && this.isDependencyField(field.fieldDependencies)) {
            dependentFields[fieldKey] = field.fieldDependencies[name];
        }
        const attributeKeys = (field.attributes && Object.keys(field.attributes)) || [];
        attributeKeys.forEach(attributeKey => {
            const attribute = field.attributes[attributeKey];
            if (!attribute || !attribute.fieldDependencies || !attribute.fieldDependencies.length) {
                return;
            }
            if (this.isDependencyField(attribute.fieldDependencies)) {
                dependentAttributes.push({
                    field: fieldKey,
                    attribute: attributeKey,
                    types: dependentFields[name]['types'] ?? []
                });
            }
        });
    }
    /**
     * Check if field is dependency
     * @param dependencies
     * @returns {boolean}
     */
    isDependencyField(dependencies) {
        const name = this.field.name || this.field.definition.name || '';
        return !!(dependencies[name] ?? false);
    }
    /**
     * Add attribute dependency
     * @param {string} fieldKey
     * @param {array} dependentFields
     * @param {object} dependentAttributes
     */
    addAttributeDependency(fieldKey, dependentFields, dependentAttributes) {
        const field = this.record.fields[fieldKey];
        const name = this.field.name || this.field.definition.name || '';
        if (fieldKey === name || !field) {
            return;
        }
        if (field.attributeDependencies && field.attributeDependencies.length && this.isDependencyAttribute(field.attributeDependencies)) {
            dependentFields[name] = field.fieldDependencies[name];
        }
        const attributeKeys = (field.attributes && Object.keys(field.attributes)) || [];
        attributeKeys.forEach(attributeKey => {
            const attribute = field.attributes[attributeKey];
            if (attribute && attribute.attributeDependencies && attribute.attributeDependencies.length) {
                const hasDependency = this.isDependencyAttribute(attribute.attributeDependencies);
                if (hasDependency) {
                    dependentAttributes.push({
                        field: fieldKey,
                        attribute: attributeKey,
                        types: (dependentFields[name] ?? {})['types'] ?? []
                    });
                }
            }
        });
    }
    /**
     * Check if attribute is dependency
     * @param {object} attributeDependencies
     * @returns {boolean}
     */
    isDependencyAttribute(attributeDependencies) {
        const parentKey = this.field.parentKey || '';
        const name = this.field.name || this.field.definition.name || '';
        return attributeDependencies.some(dependency => parentKey === dependency.field && name === dependency.attribute);
    }
    subscribeValueChanges() {
        if (this.field && this.field.formControl) {
            this.subs.push(this.field.formControl.valueChanges.subscribe(value => {
                if (!isVoid(value)) {
                    value = value.trim();
                }
                else {
                    value = '';
                }
                if (this.typeFormatter && this.field.type) {
                    value = this.toInternalFormat(this.field.type, value);
                }
                this.setFieldValue(value);
            }));
        }
    }
    toInternalFormat(fieldType, value) {
        return this.typeFormatter.toInternalFormat(fieldType, value);
    }
    setFieldValue(newValue) {
        this.field.value = newValue;
    }
    setFormControlValue(newValue) {
        if (isEqual(this.field.formControl.value, newValue)) {
            this.field.formControl.markAsPristine();
            return;
        }
        this.field.formControl.setValue(newValue);
        this.field.formControl.markAsDirty();
    }
    unsubscribeAll() {
        this.subs.forEach(sub => sub.unsubscribe());
    }
    static { this.ɵfac = function BaseFieldComponent_Factory(t) { return new (t || BaseFieldComponent)(i0.ɵɵdirectiveInject(i1.DataTypeFormatter), i0.ɵɵdirectiveInject(i2.FieldLogicManager), i0.ɵɵdirectiveInject(i3.FieldLogicDisplayManager)); }; }
    static { this.ɵcmp = /*@__PURE__*/ i0.ɵɵdefineComponent({ type: BaseFieldComponent, selectors: [["ng-component"]], inputs: { originalMode: "originalMode", field: "field", record: "record", parent: "parent", klass: "klass", mode: "mode" }, decls: 0, vars: 0, template: function BaseFieldComponent_Template(rf, ctx) { }, encapsulation: 2 }); }
}
export { BaseFieldComponent };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(BaseFieldComponent, [{
        type: Component,
        args: [{ template: '' }]
    }], function () { return [{ type: i1.DataTypeFormatter }, { type: i2.FieldLogicManager }, { type: i3.FieldLogicDisplayManager }]; }, { originalMode: [{
            type: Input
        }], field: [{
            type: Input
        }], record: [{
            type: Input
        }], parent: [{
            type: Input
        }], klass: [{
            type: Input
        }], mode: [{
            type: Input
        }] }); })();
//# sourceMappingURL=data:application/json;base64,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