/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { LogoAbstract } from '../logo/logo-abstract';
import { ready } from 'common';
import { LinkTarget } from './link-target';
export class NavbarAbstract {
    /**
     * Public API
     */
    constructor(routeConverter, moduleNavigation, preferences, language, appState, moduleNameMapper) {
        this.routeConverter = routeConverter;
        this.moduleNavigation = moduleNavigation;
        this.preferences = preferences;
        this.language = language;
        this.appState = appState;
        this.moduleNameMapper = moduleNameMapper;
        this.authenticated = true;
        this.logo = new LogoAbstract();
        this.useGroupTabs = false;
        this.globalActions = [];
        this.currentUser = {
            id: '',
            firstName: '',
            lastName: '',
        };
        this.all = {
            modules: [],
            extra: [],
        };
        this.menu = [];
    }
    /**
     * Reset menus
     */
    resetMenu() {
        this.menu = [];
        this.globalActions = [];
        this.all.modules = [];
        this.all.extra = [];
        this.current = null;
        this.currentUser = {};
    }
    /**
     * Build user action menu
     *
     * @param {[]} userActionMenu info
     * @param {object} currentUser info
     */
    buildUserActionMenu(userActionMenu, currentUser) {
        this.currentUser.id = currentUser.id;
        this.currentUser.firstName = currentUser.firstName;
        this.currentUser.lastName = currentUser.lastName;
        if (userActionMenu) {
            userActionMenu.forEach((subMenu) => {
                const name = subMenu.name;
                let url = subMenu.url;
                if (name === 'logout') {
                    return;
                }
                let target = LinkTarget.none;
                if (name === 'training') {
                    target = LinkTarget.blank;
                }
                else {
                    url = this.routeConverter.toFrontEndLink(url);
                }
                const label = this.language.getAppString(subMenu.labelKey) ?? '';
                this.globalActions.push({
                    link: {
                        url,
                        label,
                        target
                    },
                });
            });
        }
        return;
    }
    /**
     * Build navbar
     *
     * @param {object} navigation info
     * @param {object} currentUser info
     * @param {number} maxTabs to display
     */
    build(navigation, currentUser, maxTabs) {
        this.buildUserActionMenu(navigation.userActionMenu, currentUser);
        const navigationParadigm = this.preferences.getUserPreference('navigation_paradigm');
        const sort = this.preferences.getUserPreference('sort_modules_by_name') === 'on';
        if (navigationParadigm === 'm') {
            this.buildModuleNavigation(navigation, maxTabs, sort);
            return;
        }
        if (navigationParadigm === 'gm') {
            this.buildGroupedNavigation(navigation, maxTabs, sort);
            return;
        }
    }
    /**
     * Build Group tab menu
     *
     * @param {[]} items list
     * @param {object} modules info
     * @param {number} threshold limit
     * @param {object} groupedTabs info
     * @param {boolean} sort flag
     */
    buildGroupTabMenu(items, modules, threshold, groupedTabs, sort) {
        const navItems = [];
        const moreItems = [];
        if (items && items.length > 0) {
            items.forEach((module) => {
                moreItems.push(this.buildTabMenuItem(module, modules[module]));
            });
            if (sort) {
                this.sortMenuItems(moreItems);
            }
        }
        let count = 0;
        groupedTabs.forEach((groupedTab) => {
            if (count < threshold) {
                navItems.push(this.buildTabGroupedMenuItem(groupedTab.labelKey, groupedTab.modules, modules, sort));
            }
            count++;
        });
        this.menu = navItems;
        this.all.modules = moreItems;
    }
    /**
     *
     * Internal API
     *
     */
    /**
     * Build module navigation
     *
     * @param {object} navigation info
     * @param {number} maxTabs to use
     * @param {boolean} sort flag
     */
    buildModuleNavigation(navigation, maxTabs, sort) {
        if (!ready([navigation.tabs, navigation.modules])) {
            return;
        }
        this.buildTabMenu(navigation.tabs, navigation.modules, maxTabs, sort);
        this.buildSelectedModule(navigation);
    }
    /**
     * Build grouped navigation
     *
     * @param {object} navigation info
     * @param {number} maxTabs to use
     * @param {boolean} sort flag
     */
    buildGroupedNavigation(navigation, maxTabs, sort) {
        if (!ready([navigation.tabs, navigation.modules, navigation.groupedTabs])) {
            return;
        }
        this.buildGroupTabMenu(navigation.tabs, navigation.modules, maxTabs, navigation.groupedTabs, sort);
        this.buildSelectedModule(navigation);
    }
    /**
     * Build selected module
     *
     * @param {object} navigation info
     */
    buildSelectedModule(navigation) {
        const module = this.appState.getModule() ?? '';
        if (module === '' || module === 'home') {
            return;
        }
        if (!navigation.modules[module]) {
            return;
        }
        this.current = this.buildTabMenuItem(module, navigation.modules[module]);
    }
    /**
     * Build tab / module menu
     *
     * @param {[]} items list
     * @param {object} modules info
     * @param {number} threshold limit
     * @param {boolean} sort flag
     */
    buildTabMenu(items, modules, threshold, sort) {
        const navItems = [];
        const moreItems = [];
        if (!items || items.length === 0) {
            this.menu = navItems;
            this.all.modules = moreItems;
            return;
        }
        let count = 0;
        items.forEach((module) => {
            const item = this.buildTabMenuItem(module, modules[module]);
            if (module === 'home' || this.appState.getModule() === module || count > threshold) {
                moreItems.push(item);
            }
            else {
                navItems.push(item);
            }
            count++;
        });
        if (sort) {
            this.sortMenuItems(navItems);
            this.sortMenuItems(moreItems);
        }
        this.menu = navItems;
        this.all.modules = moreItems;
    }
    /**
     * Build Grouped Tab menu item
     *
     * @param {string} moduleLabel to display
     * @param {object} groupedModules list
     * @param {object} modules list
     * @param {boolean} sort flag
     *
     * @returns {object} group tab menu item
     */
    buildTabGroupedMenuItem(moduleLabel, groupedModules, modules, sort) {
        return {
            link: {
                label: this.language.getAppString(moduleLabel) || moduleLabel,
                url: '',
                route: null,
                params: null
            },
            icon: '',
            submenu: this.buildGroupedMenu(groupedModules, modules, sort)
        };
    }
    /**
     * Build Grouped menu
     *
     * @param {object} groupedModules info
     * @param {object} modules map
     * @param {boolean} sort flag
     *
     * @returns {[]} menu item array
     */
    buildGroupedMenu(groupedModules, modules, sort) {
        const groupedItems = [];
        let homeMenuItem = null;
        groupedModules.forEach((groupedModule) => {
            const module = modules[groupedModule];
            if (!module) {
                return;
            }
            const moduleMenuItem = this.buildTabMenuItem(groupedModule, module);
            if (groupedModule === 'home') {
                homeMenuItem = moduleMenuItem;
                return;
            }
            groupedItems.push(moduleMenuItem);
        });
        if (sort) {
            this.sortMenuItems(groupedItems);
        }
        if (homeMenuItem) {
            groupedItems.unshift(homeMenuItem);
        }
        return groupedItems;
    }
    /**
     * Build module menu items
     *
     * @param {string} module name
     * @param {object} moduleInfo info
     *
     * @returns {object} menuItem
     */
    buildTabMenuItem(module, moduleInfo) {
        if (moduleInfo.name) {
            module = moduleInfo.name;
        }
        const moduleRoute = this.moduleNavigation.getModuleRoute(moduleInfo);
        const appListStrings = this.language.getLanguageStrings()?.appListStrings ?? {};
        const menuItem = {
            link: {
                label: this.moduleNavigation.getModuleLabel(moduleInfo, appListStrings),
                url: moduleRoute.url,
                route: moduleRoute.route,
                params: null
            },
            icon: this.moduleNameMapper.toLegacy(module) ?? null,
            submenu: [],
            module: module ?? null,
            isGroupedMenu: false
        };
        let hasSubmenu = false;
        if (moduleInfo) {
            moduleInfo.menu.forEach((subMenu) => {
                const sublinks = subMenu.sublinks || [];
                const subMenuItem = this.buildSubMenuItem(module, subMenu, sublinks);
                menuItem.submenu.push(subMenuItem);
                if (sublinks.length > 0) {
                    hasSubmenu = true;
                }
            });
        }
        menuItem.isGroupedMenu = hasSubmenu;
        return menuItem;
    }
    buildSubMenuItem(module, subMenu, sublinks) {
        const moduleActionRoute = this.moduleNavigation.getActionRoute(subMenu);
        const subMenuItem = {
            link: {
                label: this.moduleNavigation.getActionLabel(module, subMenu, this.language.getLanguageStrings()),
                url: moduleActionRoute.url,
                route: moduleActionRoute.route,
                params: moduleActionRoute.params,
                process: moduleActionRoute.process
            },
            icon: subMenu.icon || '',
            submenu: sublinks.map((item) => this.buildSubMenuItem(module, item, [])),
        };
        return subMenuItem;
    }
    /**
     * Sort menu items by label
     *
     * @param {object} navItems to sort
     */
    sortMenuItems(navItems) {
        navItems.sort((a, b) => {
            const nameA = a.link.label.toUpperCase(); // ignore upper and lowercase
            const nameB = b.link.label.toUpperCase(); // ignore upper and lowercase
            if (nameA < nameB) {
                return -1;
            }
            if (nameA > nameB) {
                return 1;
            }
            // names must be equal
            return 0;
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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