/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Inject, Injectable, LOCALE_ID } from '@angular/core';
import { UserPreferenceStore } from '../../../store/user-preference/user-preference.store';
import { map } from 'rxjs/operators';
import { formatDate } from '@angular/common';
import { DateTime, IANAZone } from 'luxon';
import { FormControlUtils } from '../../record/field/form-control.utils';
import * as i0 from "@angular/core";
import * as i1 from "../../../store/user-preference/user-preference.store";
import * as i2 from "../../record/field/form-control.utils";
class DatetimeFormatter {
    constructor(preferences, formUtils, locale) {
        this.preferences = preferences;
        this.formUtils = formUtils;
        this.locale = locale;
        this.format$ = this.preferences.userPreferences$.pipe(map(() => {
            const date = this.getDateFormat();
            const time = this.getTimeFormat();
            return { date, time };
        }));
    }
    getDateFormat() {
        const dateFormatPreference = this.preferences.getUserPreference('date_format');
        if (dateFormatPreference) {
            return dateFormatPreference;
        }
        return this.getInternalDateFormat();
    }
    getTimeFormat() {
        const timeFormatPreference = this.preferences.getUserPreference('time_format');
        if (timeFormatPreference) {
            let format = timeFormatPreference;
            if (format.includes('aaaaaa')) {
                format = format.replace('aaaaaa', 'aaaaa\'m\'');
            }
            return format;
        }
        return this.getInternalTimeFormat();
    }
    getDateTimeFormat() {
        const dateFormat = this.getDateFormat();
        const timeFormat = this.getTimeFormat();
        return dateFormat + ' ' + timeFormat;
    }
    getInternalDateFormat() {
        return 'yyyy-MM-dd';
    }
    getInternalTimeFormat() {
        return 'HH:mm:ss';
    }
    getInternalDateTimeFormat() {
        const dateFormat = this.getInternalDateFormat();
        const timeFormat = this.getInternalTimeFormat();
        return dateFormat + ' ' + timeFormat;
    }
    getInternalFormat() {
        return this.getInternalDateTimeFormat();
    }
    getUserFormat() {
        return this.getDateTimeFormat();
    }
    /**
     * Format Internal Date to User. It assumes internal date is in GMT/UTC
     *
     * @param dateString
     * @param options
     */
    toUserFormat(dateString, options) {
        const fromFormat = (options && options.fromFormat) || this.getInternalFormat();
        const toFormat = (options && options.toFormat) || this.getUserFormat();
        if (dateString) {
            const dateTime = this.toDateTime(dateString, fromFormat, {
                zone: 'GMT'
            });
            if (!dateTime.isValid) {
                return dateString;
            }
            return formatDate(dateTime.toJSDate(), toFormat, this.locale, this.userTimeZone());
        }
        return '';
    }
    /**
     * Format User Date to Internal format. It assumes the date is in the user timezone
     *
     * @param dateString
     * @param options
     */
    toInternalFormat(dateString, options) {
        const fromFormat = (options && options.fromFormat) || this.getUserFormat();
        if (dateString) {
            let date = this.toDateTime(dateString, fromFormat, {
                zone: this.preferences.getUserPreference('timezone')
            });
            return formatDate(date.toJSDate(), this.getInternalFormat(), this.locale, 'GMT');
        }
        return '';
    }
    formatDateTime(dateString, format, fromFormat = '', locale = this.locale, timezone = '') {
        const dateTime = this.toDateTime(dateString, fromFormat);
        if (!dateTime.isValid) {
            return dateString;
        }
        return formatDate(dateTime.toJSDate(), format, locale, timezone);
    }
    toDateTime(datetimeString, fromFormat = '', options) {
        if (!datetimeString) {
            return DateTime.invalid('empty');
        }
        if (fromFormat) {
            return DateTime.fromFormat(datetimeString, fromFormat, options);
        }
        let dateTime = this.fromUserFormat(datetimeString, options);
        if (!dateTime.isValid) {
            dateTime = this.fromInternalFormat(datetimeString, options);
        }
        return dateTime;
    }
    userDateTimeFormatToStruct(datetime) {
        if (!datetime) {
            return null;
        }
        const dateTime = this.toDateTime(datetime, '', {
            zone: this.preferences.getUserPreference('timezone')
        });
        if (!dateTime.isValid) {
            return null;
        }
        return {
            date: {
                day: dateTime.day,
                month: dateTime.month,
                year: dateTime.year
            },
            time: {
                hour: dateTime.hour,
                minute: dateTime.minute,
                second: dateTime.second,
            }
        };
    }
    internalDateTimeFormatToStruct(datetime) {
        if (!datetime) {
            return null;
        }
        const dateTime = this.toDateTime(datetime, this.getInternalDateTimeFormat(), {
            zone: 'GMT'
        });
        const rezoned = dateTime.setZone(this.preferences.getUserPreference('timezone'));
        if (!dateTime.isValid) {
            return null;
        }
        return {
            date: {
                day: rezoned.day,
                month: rezoned.month,
                year: rezoned.year
            },
            time: {
                hour: rezoned.hour,
                minute: rezoned.minute,
                second: rezoned.second,
            }
        };
    }
    userDateFormatToStruct(datetime) {
        if (!datetime) {
            return null;
        }
        const dateTime = this.toDateTime(datetime, '', {
            zone: this.preferences.getUserPreference('timezone')
        });
        if (!dateTime.isValid) {
            return null;
        }
        return {
            day: dateTime.day,
            month: dateTime.month,
            year: dateTime.year
        };
    }
    dateFormatToStruct(datetime, fromFormat = '') {
        if (!datetime) {
            return null;
        }
        const dateTime = this.toDateTime(datetime, fromFormat);
        if (!dateTime.isValid) {
            return null;
        }
        return {
            day: dateTime.day,
            month: dateTime.month,
            year: dateTime.year
        };
    }
    userTimeFormatToStruct(datetime) {
        if (!datetime) {
            return null;
        }
        const dateTime = this.toDateTime(datetime, '', {
            zone: this.preferences.getUserPreference('timezone')
        });
        if (!dateTime.isValid) {
            return null;
        }
        return {
            hour: dateTime.hour,
            minute: dateTime.minute,
            second: dateTime.second
        };
    }
    fromUserFormat(datetime, options, formatOptions) {
        // ensure datetime is in user format.
        datetime = this.toUserFormat(datetime, formatOptions);
        datetime = datetime.toString();
        datetime = datetime.replace('a', 'A');
        datetime = datetime.replace('p', 'P');
        datetime = datetime.replace('m', 'M');
        let format = this.getUserFormat();
        format = format.replace('aaaaa\'m\'', 'a');
        return DateTime.fromFormat(datetime, format, options);
    }
    fromInternalFormat(datetime, options) {
        const format = this.getInternalFormat();
        return DateTime.fromFormat(datetime.toString(), format, options);
    }
    validateUserFormat(inputValue, userFormat = '') {
        const trimmedInputValue = this.formUtils.getTrimmedInputValue(inputValue);
        if (this.formUtils.isEmptyInputValue(trimmedInputValue)) {
            return false;
        }
        const dateTime = this.fromUserFormat(trimmedInputValue, {}, { fromFormat: userFormat });
        return !dateTime.isValid;
    }
    userTimeZone() {
        let userTZ = this.preferences.getUserPreference('timezone') ?? 'GMT';
        if (!userTZ) {
            userTZ = 'GMT';
        }
        const milliseconds = DateTime.now().setZone(userTZ).toMillis();
        return IANAZone.create(userTZ).formatOffset(milliseconds, 'techie');
    }
    static { this.ɵfac = function DatetimeFormatter_Factory(t) { return new (t || DatetimeFormatter)(i0.ɵɵinject(i1.UserPreferenceStore), i0.ɵɵinject(i2.FormControlUtils), i0.ɵɵinject(LOCALE_ID)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: DatetimeFormatter, factory: DatetimeFormatter.ɵfac, providedIn: 'root' }); }
}
export { DatetimeFormatter };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(DatetimeFormatter, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], function () { return [{ type: i1.UserPreferenceStore }, { type: i2.FormControlUtils }, { type: undefined, decorators: [{
                type: Inject,
                args: [LOCALE_ID]
            }] }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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