/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Injectable } from '@angular/core';
import { BehaviorSubject, of } from 'rxjs';
import { distinctUntilChanged, map, shareReplay, tap } from 'rxjs/operators';
import { EntityGQL } from '../../services/api/graphql-api/api.entity.get';
import { deepClone } from 'common';
import { AppStateStore } from '../app-state/app-state.store';
import * as i0 from "@angular/core";
import * as i1 from "../../services/api/graphql-api/api.entity.get";
import * as i2 from "../app-state/app-state.store";
const initialState = {
    module: '',
    detailView: {},
    editView: {},
    listView: {},
    search: {},
    recordView: {},
    subPanel: {},
    massUpdate: {},
    recentlyViewed: [],
    favorites: [],
    fieldActions: {}
};
const initialModuleMetadataState = {};
let internalState = deepClone(initialState);
let allModulesState = deepClone(initialModuleMetadataState);
const initialCache = {};
let cache = deepClone(initialCache);
class MetadataStore {
    constructor(recordGQL, appState) {
        this.recordGQL = recordGQL;
        this.appState = appState;
        this.typeKeys = {
            listView: 'listView',
            search: 'search',
            recordView: 'recordView',
            subPanel: 'subPanel',
            massUpdate: 'massUpdate',
            recentlyViewed: 'recentlyViewed',
            favorites: 'favorites'
        };
        this.store = new BehaviorSubject(internalState);
        this.state$ = this.store.asObservable();
        this.allModuleStore = new BehaviorSubject(allModulesState);
        this.allModulesState$ = this.allModuleStore.asObservable();
        this.resourceName = 'moduleMetadata';
        this.fieldsMetadata = {
            fields: [
                'id',
                '_id',
            ]
        };
        this.types = [
            'listView',
            'search',
            'recordView',
            'subPanel',
            'massUpdate',
            'recentlyViewed',
            'favorites'
        ];
        this.baseTypes = [
            'listView',
            'search',
            'recordView',
            'favorites'
        ];
        this.listViewColumns$ = this.state$.pipe(map(state => state.listView.fields), distinctUntilChanged());
        this.listViewLineActions$ = this.state$.pipe(map(state => state.listView.lineActions), distinctUntilChanged());
        this.listViewTableActions$ = this.state$.pipe(map(state => state.listView.tableActions), distinctUntilChanged());
        this.listMetadata$ = this.state$.pipe(map(state => state.listView), distinctUntilChanged());
        this.searchMetadata$ = this.state$.pipe(map(state => state.search), distinctUntilChanged());
        this.recordViewMetadata$ = this.state$.pipe(map(state => state.recordView), distinctUntilChanged());
        this.fieldActions$ = this.state$.pipe(map(state => state.fieldActions), distinctUntilChanged());
        this.subPanelMetadata$ = this.state$.pipe(map(state => state.subPanel), distinctUntilChanged());
        this.metadata$ = this.state$;
        this.allModuleMetadata$ = this.allModulesState$;
    }
    /**
     * Clear state
     */
    clear() {
        cache = deepClone(initialCache);
        allModulesState = deepClone(initialModuleMetadataState);
        this.updateState('', deepClone(initialState));
    }
    clearAuthBased() {
        this.clear();
    }
    /**
     * Get all metadata types
     *
     * @returns {string[]} metadata types
     */
    getMetadataTypes() {
        return this.types;
    }
    getModule() {
        return internalState.module;
    }
    get() {
        return internalState;
    }
    getModuleMeta(module) {
        const meta = allModulesState[module] ?? {};
        return { ...meta };
    }
    setModuleMetadata(module, metadata) {
        cache[module] = of(metadata).pipe(shareReplay(1));
        this.updateAllModulesState(module, metadata);
    }
    /**
     * Initial ListViewMeta load if not cached and update state.
     *
     * @param {string} moduleID to fetch
     * @param {string[]} types to fetch
     * @param useCache
     * @returns any data
     */
    reloadModuleMetadata(moduleID, types, useCache = true) {
        if (!types) {
            types = this.getMetadataTypes();
        }
        return this.getMetadata(moduleID, types, useCache).pipe(tap((metadata) => {
            this.updateAllModulesState(moduleID, metadata);
        }));
    }
    /**
     * Initial ListViewMeta load if not cached and update state.
     *
     * @param {string} moduleID to fetch
     * @param {string[]} types to fetch
     * @param useCache
     * @returns any data
     */
    load(moduleID, types, useCache = true) {
        if (!types) {
            types = this.getMetadataTypes();
        }
        return this.getMetadata(moduleID, types, useCache).pipe(tap((metadata) => {
            this.updateState(moduleID, metadata);
        }));
    }
    /**
     * Check if loaded
     */
    isCached(module) {
        return (cache[module] ?? null) !== null;
    }
    /**
     * Get empty Metadata
     */
    getEmpty() {
        return deepClone(initialState);
    }
    /**
     * Set pre-loaded navigation and cache
     */
    set(module, metadata) {
        cache[module] = of(metadata).pipe(shareReplay(1));
        this.updateState(module, metadata);
    }
    /**
     * Get ListViewMeta cached Observable or call the backend
     *
     * @param {string} module to fetch
     * @param {string[]} types to retrieve
     * @param useCache
     * @returns {object} Observable<any>
     */
    getMetadata(module, types = null, useCache = true) {
        if (cache[module] == null || useCache === false) {
            cache[module] = this.fetchMetadata(module, types).pipe(shareReplay(1));
        }
        return cache[module];
    }
    /**
     * Internal API
     */
    mapMetadata(module, data) {
        const moduleMetadata = allModulesState[module] ?? {};
        const metadata = { ...moduleMetadata };
        this.parseListViewMetadata(data, metadata);
        this.parseSearchMetadata(data, metadata);
        this.parseRecordViewMetadata(data, metadata);
        this.parseSubPanelMetadata(data, metadata);
        this.parseMassUpdateMetadata(data, metadata);
        this.parseRecentlyViewedMetadata(data, metadata);
        this.parseFavoritesMetadata(data, metadata);
        this.parseFieldViewMetada(data, metadata);
        return metadata;
    }
    /**
     * Update the state
     *
     * @param {string} module
     * @param {object} state to set
     */
    updateState(module, state) {
        this.updateAllModulesState(module, state);
        this.store.next(internalState = { ...state, module });
    }
    /**
     * Update the state
     *
     * @param {string} module
     * @param {object} state to set
     */
    updateAllModulesState(module, state) {
        if (module !== '') {
            const newState = {
                ...allModulesState
            };
            newState[module] = { ...state };
            this.allModuleStore.next(allModulesState = newState);
        }
    }
    /**
     * Fetch the Metadata from the backend
     *
     * @param {string} module to fetch
     * @param {string[]} types to retrieve
     * @returns {object} Observable<{}>
     */
    fetchMetadata(module, types) {
        const fieldsToRetrieve = {
            fields: [
                ...this.fieldsMetadata.fields,
                ...(types ?? this.baseTypes)
            ]
        };
        return this.recordGQL.fetch(this.resourceName, `/api/module-metadata/${module}`, fieldsToRetrieve)
            .pipe(map(({ data }) => {
            return this.mapMetadata(module, data.moduleMetadata);
        }));
    }
    parseListViewMetadata(data, metadata) {
        if (!data || !data.listView) {
            return;
        }
        const listViewMeta = {
            fields: [],
            bulkActions: {},
            lineActions: [],
            tableActions: [],
            chartTypes: {},
            filters: []
        };
        if (data.listView.columns) {
            data.listView.columns.forEach((field) => {
                listViewMeta.fields.push(field);
            });
        }
        const entries = {
            bulkActions: 'bulkActions',
            lineActions: 'lineActions',
            tableActions: 'tableActions',
            sidebarWidgets: 'sidebarWidgets',
            availableFilters: 'filters',
            paginationType: 'paginationType'
        };
        this.addDefinedMeta(listViewMeta, data.listView, entries);
        metadata.listView = listViewMeta;
    }
    parseFieldViewMetada(data, metadata) {
        if (!data || !data.recordView || !data.recordView.panels) {
            return;
        }
        const fieldActions = {
            recordView: {}
        };
        data.recordView.panels.forEach(panel => {
            if (panel.rows) {
                panel.rows.forEach(row => {
                    if (row.cols) {
                        row.cols.forEach(col => {
                            if (col.fieldActions && col.fieldActions.actions) {
                                Object.values(col.fieldActions.actions).forEach(action => {
                                    action['fieldName'] = col.name;
                                    const viewFieldActions = fieldActions['recordView'][col.name] ?? [];
                                    viewFieldActions.push(action);
                                    fieldActions['recordView'][col.name] = viewFieldActions;
                                });
                            }
                        });
                    }
                });
            }
        });
        metadata.fieldActions = fieldActions;
    }
    parseSearchMetadata(data, metadata) {
        if (data && data.search) {
            metadata.search = data.search;
        }
    }
    parseSubPanelMetadata(data, metadata) {
        if (data && data.subPanel) {
            metadata.subPanel = data.subPanel;
        }
    }
    parseMassUpdateMetadata(data, metadata) {
        if (data && data.massUpdate) {
            metadata.massUpdate = data.massUpdate;
        }
    }
    parseRecordViewMetadata(data, metadata) {
        if (!data || !data.recordView) {
            return;
        }
        const recordViewMeta = {
            actions: [],
            templateMeta: {},
            panels: []
        };
        const receivedMeta = data.recordView;
        const entries = {
            templateMeta: 'templateMeta',
            actions: 'actions',
            panels: 'panels',
            topWidget: 'topWidget',
            sidebarWidgets: 'sidebarWidgets',
            bottomWidgets: 'bottomWidgets',
            summaryTemplates: 'summaryTemplates',
            vardefs: 'vardefs',
            metadata: 'metadata'
        };
        this.addDefinedMeta(recordViewMeta, receivedMeta, entries);
        metadata.recordView = recordViewMeta;
    }
    parseRecentlyViewedMetadata(data, metadata) {
        if (data && data.recentlyViewed) {
            metadata.recentlyViewed = data.recentlyViewed;
        }
    }
    parseFavoritesMetadata(data, metadata) {
        if (data && data.favorites) {
            metadata.favorites = data.favorites;
        }
    }
    addDefinedMeta(metadata, received, keyMap) {
        Object.keys(keyMap).forEach(dataKey => {
            const metadataKey = keyMap[dataKey];
            if (received[dataKey]) {
                metadata[metadataKey] = received[dataKey];
            }
        });
    }
    static { this.ɵfac = function MetadataStore_Factory(t) { return new (t || MetadataStore)(i0.ɵɵinject(i1.EntityGQL), i0.ɵɵinject(i2.AppStateStore)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: MetadataStore, factory: MetadataStore.ɵfac, providedIn: 'root' }); }
}
export { MetadataStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(MetadataStore, [{
        type: Injectable,
        args: [{
                providedIn: 'root',
            }]
    }], function () { return [{ type: i1.EntityGQL }, { type: i2.AppStateStore }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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