/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { deepClone, emptyObject, PageSelection, SelectionStatus, SortDirection } from 'common';
import { BehaviorSubject, combineLatestWith, of } from 'rxjs';
import { catchError, distinctUntilChanged, map, shareReplay, take, tap } from 'rxjs/operators';
import { Injectable } from '@angular/core';
import { ListGQL } from './graphql/api.list.get';
import { SystemConfigStore } from '../system-config/system-config.store';
import { UserPreferenceStore } from '../user-preference/user-preference.store';
import { LanguageStore } from '../language/language.store';
import { MessageService } from '../../services/message/message.service';
import { LocalStorageService } from "../../services/local-storage/local-storage.service";
import * as i0 from "@angular/core";
import * as i1 from "./graphql/api.list.get";
import * as i2 from "../system-config/system-config.store";
import * as i3 from "../user-preference/user-preference.store";
import * as i4 from "../language/language.store";
import * as i5 from "../../services/message/message.service";
import * as i6 from "../../services/local-storage/local-storage.service";
const initialFilter = {
    key: 'default',
    module: 'saved-search',
    attributes: {
        contents: ''
    },
    criteria: {
        name: 'default',
        filters: {}
    }
};
const initialFilters = {
    'default': deepClone(initialFilter)
};
const initialSearchCriteria = {
    filters: {}
};
const initialListSort = {
    orderBy: '',
    sortOrder: SortDirection.DESC
};
const initialListPagination = {
    pageSize: 5,
    current: 0,
    previous: 0,
    next: 5,
    last: 0,
    total: 0,
    pageFirst: 0,
    pageLast: 0
};
const initialSelection = {
    all: false,
    status: SelectionStatus.NONE,
    selected: {},
    count: 0
};
const initialState = {
    module: '',
    records: [],
    criteria: deepClone(initialSearchCriteria),
    activeFilters: deepClone(initialFilters),
    sort: deepClone(initialListSort),
    pagination: deepClone(initialListPagination),
    selection: deepClone(initialSelection),
    openFilter: deepClone(initialFilter),
    loading: false,
    meta: {}
};
class RecordListStore {
    constructor(listGQL, configStore, preferencesStore, languageStore, message, localStorageService) {
        this.listGQL = listGQL;
        this.configStore = configStore;
        this.preferencesStore = preferencesStore;
        this.languageStore = languageStore;
        this.message = message;
        this.localStorageService = localStorageService;
        /** Internal Properties */
        this.cache$ = null;
        this.internalState = deepClone(initialState);
        this.store = new BehaviorSubject(this.internalState);
        this.state$ = this.store.asObservable();
        this.subs = [];
        this.pageKey = null;
        this.records$ = this.state$.pipe(map(state => state.records), distinctUntilChanged());
        this.criteria$ = this.state$.pipe(map(state => state.criteria), distinctUntilChanged());
        this.sort$ = this.state$.pipe(map(state => state.sort), distinctUntilChanged());
        this.pagination$ = this.state$.pipe(map(state => state.pagination), distinctUntilChanged());
        this.selection$ = this.state$.pipe(map(state => state.selection), distinctUntilChanged());
        this.selectedCount$ = this.state$.pipe(map(state => state.selection.count), distinctUntilChanged());
        this.selectedStatus$ = this.state$.pipe(map(state => state.selection.status), distinctUntilChanged());
        this.activeFilters$ = this.state$.pipe(map(state => state.activeFilters), distinctUntilChanged());
        this.openFilter$ = this.state$.pipe(map(state => state.openFilter), distinctUntilChanged());
        this.loading$ = this.state$.pipe(map(state => state.loading));
    }
    connect() {
        return this.records$;
    }
    disconnect() {
    }
    get criteria() {
        if (!this.internalState.criteria) {
            return deepClone(initialSearchCriteria);
        }
        return deepClone(this.internalState.criteria);
    }
    set criteria(criteria) {
        this.updateState({
            ...this.internalState,
            criteria
        });
    }
    get activeFilters() {
        return deepClone(this.internalState.activeFilters);
    }
    get sort() {
        if (!this.internalState.sort) {
            return deepClone(initialListSort);
        }
        return deepClone(this.internalState.sort);
    }
    set sort(sort) {
        this.updateState({
            ...this.internalState,
            sort
        });
    }
    get pagination() {
        if (!this.internalState.pagination) {
            return deepClone(initialListPagination);
        }
        return deepClone(this.internalState.pagination);
    }
    set pagination(pagination) {
        this.updateState({
            ...this.internalState,
            pagination
        });
    }
    get selection() {
        if (!this.internalState.selection) {
            return deepClone(initialSelection);
        }
        return deepClone(this.internalState.selection);
    }
    get records() {
        if (!this.internalState.records) {
            return [];
        }
        return this.internalState.records;
    }
    getModule() {
        return this.internalState.module;
    }
    getRecord(id) {
        let record = null;
        this.records.some(item => {
            if (item.id === id) {
                record = item;
                return true;
            }
        });
        return record;
    }
    /**
     * Clean destroy
     */
    destroy() {
        this.clear();
    }
    /**
     * Initial list records load if not cached and update state.
     * Returns observable to be used in resolver if needed
     *
     * @param {string} module to use
     * @param {boolean} load if to load
     * @param {string} pageSizeConfigKey string
     * @param filter
     * @param preferenceKey
     * @returns {object} Observable<any>
     */
    init(module, load = true, pageSizeConfigKey = 'list_max_entries_per_page', filter = deepClone(initialFilter), preferenceKey = '') {
        this.internalState.module = module;
        this.preferenceKey = preferenceKey;
        if (pageSizeConfigKey) {
            this.watchPageSize(pageSizeConfigKey);
        }
        this.setBaseFilter(filter);
        this.loadCurrentFilter(module);
        if (load === false) {
            return null;
        }
        return this.load();
    }
    setBaseFilter(filter) {
        this.baseFilterMap = { 'default': deepClone(filter) };
        this.baseFilter = deepClone(filter);
        this.updateState({ ...this.internalState, activeFilters: deepClone(this.baseFilterMap), openFilter: deepClone(this.baseFilter) });
    }
    /**
     * Load current filter
     * @param module
     * @protected
     */
    loadCurrentFilter(module) {
        const activeFiltersPref = this.loadPreference(module, 'current-filters') ?? this.baseFilterMap;
        if (!activeFiltersPref || emptyObject(activeFiltersPref)) {
            return;
        }
        let currentSort = this.loadPreference(module, 'current-sort');
        if (!currentSort && emptyObject(currentSort)) {
            currentSort = null;
        }
        this.setFilters(activeFiltersPref, false, currentSort);
    }
    /**
     * Set active filters
     *
     * @param {object} filters to set
     * @param {boolean} reload flag
     * @param sort
     */
    setFilters(filters, reload = true, sort = null) {
        const filterKey = Object.keys(filters).shift();
        const filter = filters[filterKey];
        this.updateState({ ...this.internalState, activeFilters: deepClone(filters), openFilter: deepClone(filter) });
        if (filter.criteria) {
            let orderBy = filter.criteria.orderBy ?? '';
            const sortOrder = filter.criteria.sortOrder ?? 'desc';
            let direction = this.mapSortOrder(sortOrder);
            if (sort !== null) {
                orderBy = sort.orderBy;
                direction = sort.sortOrder;
            }
            this.updateSorting(orderBy, direction, false);
            this.updateSortLocalStorage();
            this.updateSearchCriteria(filter.criteria, reload);
        }
        this.updateFilterLocalStorage();
    }
    updateFilterLocalStorage() {
        const module = this.internalState.module;
        this.savePreference(module, 'current-filters', this.internalState.activeFilters);
    }
    updateSortLocalStorage() {
        const module = this.internalState.module;
        this.savePreference(module, 'current-sort', this.sort);
    }
    updatePaginationLocalStorage() {
        if (this.pageKey === null) {
            return;
        }
        const module = this.internalState.module;
        const key = module + '-' + this.pageKey + '-' + 'current-pagination';
        this.localStorageService.set(key, this.pagination);
    }
    /**
     * Load / reload records using current pagination and criteria
     *
     * @param {boolean} useCache if to use cache
     * @returns {object} Observable<ListViewState>
     */
    load(useCache = true) {
        this.updateState({
            ...this.internalState,
            loading: true
        });
        return this.getRecords(this.internalState.module, this.internalState.criteria, this.internalState.sort, this.internalState.pagination, useCache).pipe(catchError(() => {
            this.message.addDangerMessageByKey('LBL_GET_RECORD_LIST_ERROR');
            return of({
                records: [],
                criteria: deepClone(initialSearchCriteria),
                sort: deepClone(initialListSort),
                pagination: deepClone(initialListPagination),
                openFilter: deepClone(this.baseFilter),
                activeFilters: deepClone(this.baseFilterMap),
                selection: deepClone(initialSelection),
                meta: {}
            });
        }), tap((data) => {
            this.calculatePageCount(data.records, data.pagination);
            this.updateState({
                ...this.internalState,
                records: data.records,
                pagination: data.pagination,
                loading: false,
                meta: data.meta ?? {}
            });
        }));
    }
    /**
     * Update the search criteria
     *
     * @param {object} criteria to set
     * @param {boolean} reload flag
     */
    updateSearchCriteria(criteria, reload = true) {
        this.updateState({ ...this.internalState, criteria });
        if (reload) {
            this.updateSelection(SelectionStatus.NONE);
            // Reset pagination to default first page
            this.resetPagination();
        }
    }
    /**
     * Reset search criteria
     * @param {boolean} reload flag
     */
    resetSearchCriteria(reload = true) {
        this.updateSearchCriteria(deepClone(initialSearchCriteria), reload);
    }
    /**
     * Update current list view sorting
     *
     * @param {string} orderBy to set
     * @param {string} sortOrder to set
     * @param {boolean} reload flag
     */
    updateSorting(orderBy, sortOrder, reload = true) {
        if (sortOrder === SortDirection.NONE) {
            orderBy = '';
            sortOrder = SortDirection.DESC;
        }
        const sort = { orderBy, sortOrder };
        this.updateState({ ...this.internalState, sort });
        if (reload) {
            this.load(false).pipe(take(1)).subscribe();
        }
    }
    /**
     * Map sort order to SortDirection enum
     * @param {string} sortOrder to map
     * @returns {string} SortDirection
     */
    mapSortOrder(sortOrder) {
        let direction = SortDirection.NONE;
        const sort = sortOrder.toLowerCase();
        if (sort === 'asc') {
            direction = SortDirection.ASC;
        }
        else if (sort === 'desc') {
            direction = SortDirection.DESC;
        }
        return direction;
    }
    /**
     * Update the pagination
     *
     * @param {number} current to set
     */
    updatePagination(current) {
        const pagination = { ...this.internalState.pagination, current };
        this.updateState({ ...this.internalState, pagination });
        this.load(false).pipe(take(1), tap(() => this.updatePaginationLocalStorage())).subscribe();
    }
    setPagination(current) {
        const pagination = { ...this.internalState.pagination, current };
        this.updateState({ ...this.internalState, pagination });
        return this.load(false).pipe(take(1), tap(() => this.updatePaginationLocalStorage()));
    }
    /**
     * Set open filters
     *
     * @param {object} filter to set
     */
    setOpenFilter(filter) {
        this.updateState({ ...this.internalState, openFilter: deepClone(filter) });
    }
    /**
     * Reset active filters
     *
     * @param {boolean} reload flag
     */
    resetFilters(reload = true) {
        this.updateState({
            ...this.internalState,
            activeFilters: deepClone(this.baseFilterMap),
            openFilter: deepClone(this.baseFilter),
        });
        this.clearSort();
        this.updateSortLocalStorage();
        this.updateFilterLocalStorage();
        this.updateSearchCriteria(this.baseFilter.criteria, reload);
    }
    /**
     * Save ui user preference
     * @param module
     * @param storageKey
     * @param value
     * @protected
     */
    savePreference(module, storageKey, value) {
        const preferenceKey = this.preferenceKey ?? null;
        if (!preferenceKey) {
            return null;
        }
        const key = `${preferenceKey}${storageKey}`;
        this.preferencesStore.setUi(module, key, value);
    }
    /**
     * Load ui user preference
     * @param module
     * @param storageKey
     * @protected
     */
    loadPreference(module, storageKey) {
        const preferenceKey = this.preferenceKey ?? null;
        if (!preferenceKey) {
            return null;
        }
        const key = `${preferenceKey}${storageKey}`;
        return this.preferencesStore.getUi(module, key);
    }
    /**
     * Reset/Clear the pagination
     */
    resetPagination() {
        this.updatePagination(0);
    }
    /**
     * Clear observable cache
     */
    clear() {
        this.cache$ = null;
        this.store.unsubscribe();
        this.preferencesSub.unsubscribe();
    }
    clearAuthBased() {
        this.clear();
    }
    /**
     * Selection public api
     */
    getSelectionStatus() {
        return this.selectedStatus$;
    }
    getSelectedCount() {
        return this.selectedCount$;
    }
    updateSelection(state) {
        if (state === SelectionStatus.NONE) {
            this.clearSelection();
            return;
        }
        if (state === SelectionStatus.ALL) {
            this.selectAll();
            return;
        }
        if (state === SelectionStatus.PAGE) {
            this.selectPage();
            return;
        }
    }
    clearSelection() {
        this.updateState({
            ...this.internalState,
            selection: deepClone(initialSelection)
        });
    }
    clearSort() {
        this.updateState({
            ...this.internalState,
            sort: deepClone(initialListSort)
        });
    }
    selectAll() {
        const total = this.internalState.pagination.total;
        this.updateState({
            ...this.internalState,
            selection: {
                all: true,
                status: SelectionStatus.ALL,
                selected: {},
                count: total
            }
        });
    }
    selectPage() {
        const selected = { ...this.internalState.selection.selected };
        if (this.internalState.records && this.internalState.records.length) {
            this.internalState.records.forEach(value => {
                if (value && value.id) {
                    selected[value.id] = value.id;
                }
            });
        }
        this.updateState({
            ...this.internalState,
            selection: {
                all: false,
                status: SelectionStatus.SOME,
                selected,
                count: Object.keys(selected).length
            }
        });
    }
    toggleSelection(id) {
        const selection = deepClone(this.internalState.selection);
        if (selection.selected[id]) {
            delete selection.selected[id];
        }
        else {
            selection.selected[id] = id;
        }
        selection.count = Object.keys(selection.selected).length;
        if (selection.count === 0) {
            selection.status = SelectionStatus.NONE;
        }
        else {
            selection.status = SelectionStatus.SOME;
        }
        this.updateState({
            ...this.internalState,
            selection
        });
    }
    /**
     * Pagination Public API
     */
    getPaginationCount() {
        return this.pagination$.pipe(map(pagination => ({
            pageFirst: pagination.pageFirst,
            pageLast: pagination.pageLast,
            total: pagination.total
        })), distinctUntilChanged());
    }
    getPagination() {
        return this.store.value.pagination;
    }
    getMeta() {
        return this.store.value.meta;
    }
    changePage(page) {
        let pageToLoad = 0;
        const pageMap = {};
        pageMap[PageSelection.FIRST] = 0;
        pageMap[PageSelection.PREVIOUS] = this.internalState.pagination.previous;
        pageMap[PageSelection.NEXT] = this.internalState.pagination.next;
        pageMap[PageSelection.LAST] = this.internalState.pagination.last;
        if (page in pageMap && pageMap[page] >= 0) {
            pageToLoad = pageMap[page];
            if (Number(pageToLoad) > this.internalState.pagination.last) {
                return;
            }
            if (pageToLoad < 0) {
                return;
            }
            this.updatePagination(pageToLoad);
        }
    }
    setPage(page, isPaginationLoadMore) {
        let pageToLoad = 0;
        const pageMap = {};
        pageMap[PageSelection.FIRST] = 0;
        pageMap[PageSelection.PREVIOUS] = this.internalState.pagination.previous;
        pageMap[PageSelection.NEXT] = this.internalState.pagination.next;
        pageMap[PageSelection.LAST] = this.internalState.pagination.last;
        if (page in pageMap && pageMap[page] >= 0) {
            pageToLoad = pageMap[page];
            if (Number(pageToLoad) > this.internalState.pagination.last) {
                return of({});
            }
            if (pageToLoad < 0) {
                return of({});
            }
            if (isPaginationLoadMore) {
                pageToLoad = 0;
            }
            return this.setPagination(pageToLoad);
        }
        return of({});
    }
    /**
     * Set Pagination page size
     *
     * @param {number} pageSize to set
     */
    setPageSize(pageSize) {
        const pagination = { ...this.internalState.pagination, pageSize };
        this.updateState({ ...this.internalState, pagination });
    }
    /**
     * Get Pagination page size
     */
    getPageSize() {
        return this?.internalState?.pagination?.pageSize ?? 10;
    }
    /**
     * Internal API
     */
    /**
     * Subscribe to page size changes
     *
     * @param {string} pageSizeConfigKey key
     */
    watchPageSize(pageSizeConfigKey) {
        const pageSizePreference = this.preferencesStore.getUserPreference(pageSizeConfigKey);
        const pageSizeConfig = this.configStore.getConfigValue(pageSizeConfigKey);
        this.determinePageSize(pageSizePreference, pageSizeConfig);
        this.preferencesSub = this.configStore.configs$.pipe(combineLatestWith(this.preferencesStore.userPreferences$), tap(([configs, preferences]) => {
            const key = pageSizeConfigKey;
            const sizePreference = (preferences && preferences[key]) || null;
            const sizeConfig = (configs && configs[key] && configs[key].value) || null;
            this.determinePageSize(sizePreference, sizeConfig);
        })).subscribe();
    }
    /**
     * Determine page size to use
     *
     * @param {} pageSizePreference to use
     * @param {string} pageSizeConfig to use
     */
    determinePageSize(pageSizePreference, pageSizeConfig) {
        let size = 20;
        if (pageSizePreference) {
            size = pageSizePreference;
        }
        else if (pageSizeConfig) {
            size = parseInt(pageSizeConfig, 10);
        }
        this.setPageSize(size);
    }
    /**
     * Update the state
     *
     * @param {object} state to set
     */
    updateState(state) {
        this.store.next(this.internalState = state);
    }
    /**
     * Calculate page count
     *
     * @param {object} records list
     * @param {object} pagination info
     */
    calculatePageCount(records, pagination) {
        const recordCount = (records && records.length) || 0;
        let pageFirst = 0;
        let pageLast = 0;
        if (recordCount > 0) {
            pageFirst = pagination.current + 1;
            pageLast = pagination.current + recordCount;
        }
        pagination.pageFirst = pageFirst;
        pagination.pageLast = pageLast;
    }
    /**
     * Get records cached Observable or call the backend
     *
     * @param {string} module to use
     * @param {object} criteria to use
     * @param {object} sort to use
     * @param {object} pagination to use
     * @param {boolean} useCache if to use cache
     * @returns {object} Observable<any>
     */
    getRecords(module, criteria, sort, pagination, useCache = true) {
        if (this.cache$ == null || useCache === false) {
            this.cache$ = this.listGQL.get(module, criteria, sort, pagination).pipe(shareReplay(1));
        }
        return this.cache$;
    }
    static { this.ɵfac = function RecordListStore_Factory(t) { return new (t || RecordListStore)(i0.ɵɵinject(i1.ListGQL), i0.ɵɵinject(i2.SystemConfigStore), i0.ɵɵinject(i3.UserPreferenceStore), i0.ɵɵinject(i4.LanguageStore), i0.ɵɵinject(i5.MessageService), i0.ɵɵinject(i6.LocalStorageService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: RecordListStore, factory: RecordListStore.ɵfac }); }
}
export { RecordListStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(RecordListStore, [{
        type: Injectable
    }], function () { return [{ type: i1.ListGQL }, { type: i2.SystemConfigStore }, { type: i3.UserPreferenceStore }, { type: i4.LanguageStore }, { type: i5.MessageService }, { type: i6.LocalStorageService }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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