/**
 * SuiteCRM is a customer relationship management program developed by SalesAgility Ltd.
 * Copyright (C) 2024 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SALESAGILITY, SALESAGILITY DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Supercharged by SuiteCRM" logo. If the display of the logos is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Supercharged by SuiteCRM".
 */
import { Injectable } from '@angular/core';
import { deepClone, emptyObject } from 'common';
import { BehaviorSubject, of } from "rxjs";
import { distinctUntilChanged, map, shareReplay } from "rxjs/operators";
import { isArray, union } from "lodash-es";
import { SystemConfigStore } from "../../../../store/system-config/system-config.store";
import { UserPreferenceStore } from "../../../../store/user-preference/user-preference.store";
import { RecordListStoreFactory } from "../../../../store/record-list/record-list.store.factory";
import { LocalStorageService } from "../../../../services/local-storage/local-storage.service";
import { RecordPaginationService } from "./record-pagination.service";
import * as i0 from "@angular/core";
import * as i1 from "../../../../store/user-preference/user-preference.store";
import * as i2 from "../../../../store/system-config/system-config.store";
import * as i3 from "../../../../store/record-list/record-list.store.factory";
import * as i4 from "../../../../services/local-storage/local-storage.service";
import * as i5 from "./record-pagination.service";
const initialState = {
    paginationEnabled: false,
    recordIds: null,
    pagination: null
};
class RecordPaginationStore {
    constructor(preferences, systemConfigStore, listStoreFactory, localStorageService, recordPaginationService) {
        this.preferences = preferences;
        this.systemConfigStore = systemConfigStore;
        this.listStoreFactory = listStoreFactory;
        this.localStorageService = localStorageService;
        this.recordPaginationService = recordPaginationService;
        this.internalState = deepClone(initialState);
        this.cache$ = null;
        this.store = new BehaviorSubject(this.internalState);
        this.state$ = this.store.asObservable();
        this.subs = [];
        this.recordListStore = this.listStoreFactory.create();
        this.recordIds$ = this.state$.pipe(map(state => state.recordIds), distinctUntilChanged());
        this.pagination$ = this.state$.pipe(map(state => state.pagination), distinctUntilChanged());
        this.paginationEnabled$ = this.state$.pipe(map(state => state.paginationEnabled), distinctUntilChanged());
    }
    clear() {
        this.cache$ = null;
        this.updateState(deepClone(initialState));
        this.subs.forEach(sub => sub.unsubscribe());
    }
    init() {
        const module = this.getModule();
        this.recordListStore.init(module, false);
        this.enableRecordPagination();
        this.loadDataLocalStorage();
        this.loadCurrentPagination(module);
        this.loadCurrentSort(module);
        this.loadCurrentFilter(module);
    }
    enableRecordPagination() {
        let isEnabled = this.systemConfigStore.getConfigValue('enable_record_pagination');
        if (isEnabled === "") {
            isEnabled = false;
        }
        this.updateState({ ...this.internalState, paginationEnabled: !!(isEnabled ?? false) });
    }
    loadDataLocalStorage() {
        const data = this.getRecordListPreference();
        this.updateState({ ...this.internalState, recordIds: data?.recordIds, pagination: data?.pagination });
    }
    getRecordListPreference() {
        const module = this.getModule();
        const data = this.loadPreference(module, 'current-record-pagination');
        this.checkPaginationExist(data);
        if (!isArray(data.recordIds) || !data.recordIds || !data.recordIds.length) {
            return null;
        }
        return data;
    }
    checkPaginationExist(data) {
        const module = this.getModule();
        const hasPagination = this.loadPreference(module, 'current-pagination', 'listview');
        if (!hasPagination) {
            this.recordListStore.pagination = data.pagination;
        }
    }
    loadPreference(module, storageKey, pageKey) {
        if (!pageKey) {
            return this.preferences.getUi(module, this.getPreferenceKey(storageKey));
        }
        return this.preferences.getUi(module, (pageKey + '-' + storageKey));
    }
    getPreferenceKey(storageKey) {
        return 'recordview-' + storageKey;
    }
    loadCurrentPagination(module) {
        const key = module + '-' + 'listview-current-pagination';
        const currentPagination = this.localStorageService.get(key);
        if (!currentPagination || emptyObject(currentPagination)) {
            return;
        }
        this.recordListStore.pagination = currentPagination;
    }
    loadCurrentSort(module) {
        const currentSort = this.loadPreference(module, 'current-sort', 'listview');
        if (!currentSort || emptyObject(currentSort)) {
            return;
        }
        this.recordListStore.sort = currentSort;
    }
    loadCurrentFilter(module) {
        const activeFiltersPref = this.loadPreference(module, 'current-filters', 'listview') ?? {};
        if (!activeFiltersPref || emptyObject(activeFiltersPref)) {
            return;
        }
        let currentSort = this.loadPreference(module, 'current-sort', 'listview');
        if (!currentSort && emptyObject(currentSort)) {
            currentSort = null;
        }
        this.setFilters(activeFiltersPref, false, currentSort);
    }
    setFilters(filters, reload = true, sort = null) {
        const filterKey = Object.keys(filters).shift();
        const filter = filters[filterKey];
        this.recordListStore.setFilters(filters, reload, sort);
        if (filter.criteria) {
            let orderBy = filter.criteria.orderBy ?? '';
            const sortOrder = filter.criteria.sortOrder ?? '';
            let direction = this.recordListStore.mapSortOrder(sortOrder);
            if (sort !== null) {
                orderBy = sort.orderBy;
                direction = sort.sortOrder;
            }
            this.recordListStore.updateSorting(orderBy, direction, false);
        }
        this.updateSearchCriteria(filters, reload);
    }
    updateSearchCriteria(filters, reload = true) {
        let criteria = this.mergeCriteria(filters);
        this.recordListStore.updateSearchCriteria(criteria, reload);
    }
    mergeCriteria(filters) {
        let criteria = {};
        const keys = Object.keys(filters ?? {}) ?? [];
        keys.forEach(key => {
            const filter = filters[key] ?? null;
            const filterCriteria = filter?.criteria ?? null;
            const filterCriteriaKeys = Object.keys(filterCriteria?.filters ?? {});
            if (filterCriteria === null || (filterCriteriaKeys && !filterCriteriaKeys.length)) {
                return;
            }
            if (emptyObject(criteria)) {
                criteria = deepClone(filterCriteria);
                return;
            }
            filterCriteriaKeys.forEach(criteriaKey => {
                const filterCriteriaContent = filterCriteria?.filters[criteriaKey] ?? null;
                const criteriaContent = criteria?.filters[criteriaKey] ?? null;
                if (!filterCriteriaContent) {
                    return;
                }
                const criteriaOperator = criteriaContent?.operator ?? null;
                if (!criteriaContent || !criteriaOperator) {
                    criteria.filters[criteriaKey] = deepClone(filterCriteriaContent);
                    return;
                }
                const filterCriteriaOperator = filterCriteriaContent?.operator ?? null;
                if (filterCriteriaOperator !== criteriaOperator || filterCriteriaOperator !== '=') {
                    delete criteria.filters[criteriaKey];
                    return;
                }
                criteriaContent.values = union(criteriaContent.values ?? [], filterCriteriaContent.values ?? []);
            });
        });
        return criteria;
    }
    getModule() {
        return this.recordPaginationService.getModule();
    }
    getCurrentPage() {
        const pageSize = this.internalState.pagination?.pageSize;
        const pageLast = this.internalState.pagination?.pageLast;
        const currentPage = Math.ceil(pageLast / pageSize);
        return currentPage;
    }
    getPageSize() {
        return this.internalState.pagination?.pageSize;
    }
    getRecordsCount() {
        return this.internalState.pagination?.total;
    }
    updateState(state) {
        this.store.next(this.internalState = state);
    }
    set(state) {
        this.cache$ = of(state).pipe(shareReplay(1));
        this.updateState(state);
    }
    isCached() {
        return this.cache$ !== null;
    }
    static { this.ɵfac = function RecordPaginationStore_Factory(t) { return new (t || RecordPaginationStore)(i0.ɵɵinject(i1.UserPreferenceStore), i0.ɵɵinject(i2.SystemConfigStore), i0.ɵɵinject(i3.RecordListStoreFactory), i0.ɵɵinject(i4.LocalStorageService), i0.ɵɵinject(i5.RecordPaginationService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: RecordPaginationStore, factory: RecordPaginationStore.ɵfac }); }
}
export { RecordPaginationStore };
(function () { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(RecordPaginationStore, [{
        type: Injectable
    }], function () { return [{ type: i1.UserPreferenceStore }, { type: i2.SystemConfigStore }, { type: i3.RecordListStoreFactory }, { type: i4.LocalStorageService }, { type: i5.RecordPaginationService }]; }, null); })();
//# sourceMappingURL=data:application/json;base64,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